/* 
 *  © 2024 Appstock 
 *  */

package com.appstock.demo.java.activities.ads.admob;

import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.google.android.gms.ads.AdListener;
import com.google.android.gms.ads.AdLoader;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.nativead.NativeAd;
import com.google.android.gms.ads.nativead.NativeAdOptions;
import com.google.android.gms.ads.nativead.NativeAdView;
import com.appstock.demo.java.activities.BaseAdActivity;
import com.appstock.demo.java.databinding.ViewNativeAdAdMobBinding;
import com.appstock.sdk.admob.AppstockGadMediationAdapter;
import com.appstock.sdk.api.adunits.nativead.request.*;
import com.appstock.sdk.api.adunits.nativead.response.NativeEventTracker;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

public class AdMobNativeActivity extends BaseAdActivity {

    private static final String ADMOB_AD_UNIT_ID = "ca-app-pub-2844566227051243/6967515093";

    private AdLoader adLoader;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        createAd();
    }


    private void createAd() {
        // 1. Create native ad config
        AppstockNativeAdConfig nativeConfig = createNativeConfig();
        Bundle extrasBundle = nativeConfig.toGmaBundle();

        // 2.  Create ad request with AppstockGadMediationAdapter and native ad config
        AdRequest adRequest = new AdRequest.Builder()
                .addNetworkExtrasBundle(AppstockGadMediationAdapter.class, extrasBundle)
                .build();

        // 3. Configure ad loader
        adLoader = new AdLoader.Builder(this, ADMOB_AD_UNIT_ID)
                .forNativeAd((ad) -> {
                    // 5. Create native view
                    createCustomView(ad);
                })
                .withAdListener(createListener())
                .withNativeAdOptions(new NativeAdOptions.Builder().build())
                .build();

        // 4. Load ad
        adLoader.loadAd(adRequest);
    }


    private AppstockNativeAdConfig createNativeConfig() {
        ArrayList<NativeEventTracker.EventTrackingMethod> eventTrackingMethods = new ArrayList<>(
                Arrays.asList(
                        NativeEventTracker.EventTrackingMethod.IMAGE,
                        NativeEventTracker.EventTrackingMethod.JS
                )
        );
        NativeEventTracker eventTracker = new NativeEventTracker(
                NativeEventTracker.EventType.IMPRESSION,
                eventTrackingMethods
        );

        NativeTitleAsset title = new NativeTitleAsset();
        title.setLength(90);
        title.setRequired(true);

        NativeImageAsset icon = new NativeImageAsset(20, 20, 20, 20);
        icon.setImageType(NativeImageAsset.ImageType.ICON);
        icon.setRequired(true);

        NativeImageAsset mainImage = new NativeImageAsset(200, 200, 200, 200);
        mainImage.setImageType(NativeImageAsset.ImageType.MAIN);
        mainImage.setRequired(true);

        NativeDataAsset sponsored = new NativeDataAsset();
        sponsored.setLen(90);
        sponsored.setDataType(NativeDataAsset.DataType.SPONSORED);
        sponsored.setRequired(true);


        NativeDataAsset description = new NativeDataAsset();
        description.setDataType(NativeDataAsset.DataType.DESC);
        description.setRequired(true);

        NativeDataAsset ctaText = new NativeDataAsset();
        ctaText.setDataType(NativeDataAsset.DataType.CTATEXT);
        ctaText.setRequired(true);

        List<NativeAsset> assets = Arrays.asList(
                title,
                icon,
                mainImage,
                sponsored,
                description,
                ctaText
        );

        return new AppstockNativeAdConfig.Builder()
                .setContextType(NativeContextType.SOCIAL_CENTRIC)
                .setPlacementType(NativePlacementType.CONTENT_FEED)
                .setContextSubType(NativeContextSubtype.GENERAL_SOCIAL)
                .setNativeEventTrackers(Collections.singletonList(eventTracker))
                .setNativeAssets(assets)
                .build();
    }


    private void createCustomView(NativeAd ad) {
        ViewNativeAdAdMobBinding binding = ViewNativeAdAdMobBinding.inflate(LayoutInflater.from(this));

        binding.tvHeadline.setText(ad.getHeadline());
        binding.tvBody.setText(ad.getBody());
        NativeAd.Image icon = ad.getIcon();
        if (icon != null) {
            binding.imgIco.setImageDrawable(icon.getDrawable());
        }
        if (!ad.getImages().isEmpty()) {
            binding.imgMedia.setMediaContent(ad.getMediaContent());
        }

        NativeAdView container = binding.viewNativeWrapper;
        container.setHeadlineView(binding.tvHeadline);
        container.setBodyView(binding.tvBody);
        container.setIconView(binding.imgIco);
        container.setMediaView(binding.imgMedia);
        container.setNativeAd(ad);

        getContainerForAd().addView(binding.getRoot());
    }


    private static AdListener createListener() {
        return new AdListener() {
            @Override
            public void onAdLoaded() {
                Log.d(TAG, "Ad loaded successfully");
            }

            @Override
            public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
                Log.e(TAG, "Ad failed to load: " + loadAdError);
            }
        };
    }

}
