/* 
 *  © 2024 Appstock 
 *  */

package com.appstock.demo.java.activities.ads.inapp;

import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.TextView;
import androidx.annotation.Nullable;
import com.appstock.demo.java.R;
import com.appstock.demo.java.activities.BaseAdActivity;
import com.appstock.demo.java.utils.ImageUtils;
import com.appstock.sdk.api.adunits.AppstockNativeAdUnit;
import com.appstock.sdk.api.adunits.AppstockNativeAdUnitEventListener;
import com.appstock.sdk.api.adunits.nativead.request.*;
import com.appstock.sdk.api.adunits.nativead.response.NativeEventTracker;
import com.appstock.sdk.api.AdInfo;
import com.appstock.sdk.api.adunits.nativead.response.AppstockNativeAd;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

public class AppstockNativeActivity extends BaseAdActivity {

    private static final String PLACEMENT_ID = "9";

    private AppstockNativeAdUnit adUnit;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        createAd();
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (adUnit != null) {
            adUnit.destroy();
        }
    }


    private void createAd() {
        // 1. Create AppstockNativeAdUnit
        adUnit = new AppstockNativeAdUnit();

        // 2. Configure ad unit with native config
        adUnit.setPlacementId(PLACEMENT_ID);
        adUnit.setNativeAdConfig(createNativeConfig());

        // 3. Load ad
        adUnit.loadAd((result) -> {
            AppstockNativeAd nativeAd = result.getNativeAd();
            if (nativeAd == null) {
                Log.e("AdExample", "Native ad is null: " + result.getStatus());
                return;
            }

            Log.d(TAG, "Native ad loaded successfully");
            // 4. Create native view
            createNativeView(nativeAd);
        });
    }

    private AppstockNativeAdConfig createNativeConfig() {
        ArrayList<NativeEventTracker.EventTrackingMethod> eventTrackingMethods = new ArrayList<>(
                Arrays.asList(
                        NativeEventTracker.EventTrackingMethod.IMAGE,
                        NativeEventTracker.EventTrackingMethod.JS
                )
        );
        NativeEventTracker eventTracker = new NativeEventTracker(
                NativeEventTracker.EventType.IMPRESSION,
                eventTrackingMethods
        );

        NativeTitleAsset title = new NativeTitleAsset();
        title.setLength(90);
        title.setRequired(true);

        NativeImageAsset icon = new NativeImageAsset(20, 20, 20, 20);
        icon.setImageType(NativeImageAsset.ImageType.ICON);
        icon.setRequired(true);

        NativeImageAsset mainImage = new NativeImageAsset(200, 200, 200, 200);
        mainImage.setImageType(NativeImageAsset.ImageType.MAIN);
        mainImage.setRequired(true);

        NativeDataAsset sponsored = new NativeDataAsset();
        sponsored.setLen(90);
        sponsored.setDataType(NativeDataAsset.DataType.SPONSORED);
        sponsored.setRequired(true);


        NativeDataAsset description = new NativeDataAsset();
        description.setDataType(NativeDataAsset.DataType.DESC);
        description.setRequired(true);

        NativeDataAsset ctaText = new NativeDataAsset();
        ctaText.setDataType(NativeDataAsset.DataType.CTATEXT);
        ctaText.setRequired(true);

        List<NativeAsset> assets = Arrays.asList(
                title,
                icon,
                mainImage,
                sponsored,
                description,
                ctaText
        );

        return new AppstockNativeAdConfig.Builder()
                .setContextType(NativeContextType.SOCIAL_CENTRIC)
                .setPlacementType(NativePlacementType.CONTENT_FEED)
                .setContextSubType(NativeContextSubtype.GENERAL_SOCIAL)
                .setNativeEventTrackers(Collections.singletonList(eventTracker))
                .setNativeAssets(assets)
                .build();
    }

    private void createNativeView(AppstockNativeAd ad) {
        View nativeContainer = View.inflate(this, R.layout.layout_native, null);

        ImageView icon = nativeContainer.findViewById(R.id.imgIcon);
        ImageUtils.download(ad.getIconUrl(), icon);

        TextView title = nativeContainer.findViewById(R.id.tvTitle);
        title.setText(ad.getTitle());

        ImageView image = nativeContainer.findViewById(R.id.imgImage);
        ImageUtils.download(ad.getImageUrl(), image);

        TextView description = nativeContainer.findViewById(R.id.tvDesc);
        description.setText(ad.getDescription());

        Button cta = nativeContainer.findViewById(R.id.btnCta);
        cta.setText(ad.getCallToAction());
        getContainerForAd().addView(nativeContainer);

        ad.registerView(nativeContainer, Arrays.asList(icon, title, image, description, cta), createListener());
    }

    private static AppstockNativeAdUnitEventListener createListener() {
        return new AppstockNativeAdUnitEventListener() {
            @Override
            public void onAdImpression() {
                // Called when ad displayed
                Log.d(TAG, "Ad displayed on the screen");
            }

            @Override
            public void onAdClicked() {
                // Called when ad clicked
                Log.d(TAG, "Ad clicked");
            }

            @Override
            public void onAdExpired() {
                // Called when ad expired
                Log.d(TAG, "Ad expired");
            }
        };
    }

}
