/* 
 *  © 2024 Appstock 
 *  */

package com.appstock.demo.kotlin.activities.ads.admob

import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import com.google.android.gms.ads.AdListener
import com.google.android.gms.ads.AdLoader
import com.google.android.gms.ads.AdRequest
import com.google.android.gms.ads.LoadAdError
import com.google.android.gms.ads.nativead.NativeAd
import com.google.android.gms.ads.nativead.NativeAdOptions
import com.google.android.gms.ads.nativead.NativeAdView
import com.appstock.demo.kotlin.activities.BaseAdActivity
import com.appstock.demo.kotlin.databinding.ViewNativeAdAdMobBinding
import com.appstock.sdk.admob.AppstockGadMediationAdapter
import com.appstock.sdk.api.adunits.nativead.request.*
import com.appstock.sdk.api.adunits.nativead.response.NativeEventTracker
import java.util.*

class AdMobNativeActivity : BaseAdActivity() {
    private var adLoader: AdLoader? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        createAd()
    }


    private fun createAd() {
        // 1. Create native ad config
        val nativeConfig = createNativeConfig()
        val extrasBundle = nativeConfig.toGmaBundle()

        // 2.  Create ad request with AppstockGadMediationAdapter and native ad config
        val adRequest = AdRequest.Builder()
            .addNetworkExtrasBundle(AppstockGadMediationAdapter::class.java, extrasBundle)
            .build()

        // 3. Configure ad loader
        adLoader = AdLoader.Builder(this, ADMOB_AD_UNIT_ID)
            .forNativeAd { ad: NativeAd ->
                createCustomView(ad)
            }
            .withAdListener(createListener())
            .withNativeAdOptions(NativeAdOptions.Builder().build())
            .build()

        // 4. Load ad
        adLoader!!.loadAd(adRequest)
    }


    private fun createNativeConfig(): AppstockNativeAdConfig {
        val eventTrackingMethods = ArrayList(
            Arrays.asList(
                NativeEventTracker.EventTrackingMethod.IMAGE,
                NativeEventTracker.EventTrackingMethod.JS
            )
        )
        val eventTracker = NativeEventTracker(
            NativeEventTracker.EventType.IMPRESSION,
            eventTrackingMethods
        )

        val title = NativeTitleAsset()
        title.setLength(90)
        title.isRequired = true

        val icon = NativeImageAsset(20, 20, 20, 20)
        icon.imageType = NativeImageAsset.ImageType.ICON
        icon.isRequired = true

        val mainImage = NativeImageAsset(200, 200, 200, 200)
        mainImage.imageType = NativeImageAsset.ImageType.MAIN
        mainImage.isRequired = true

        val sponsored = NativeDataAsset()
        sponsored.len = 90
        sponsored.dataType = NativeDataAsset.DataType.SPONSORED
        sponsored.isRequired = true


        val description = NativeDataAsset()
        description.dataType = NativeDataAsset.DataType.DESC
        description.isRequired = true

        val ctaText = NativeDataAsset()
        ctaText.dataType = NativeDataAsset.DataType.CTATEXT
        ctaText.isRequired = true

        val assets = Arrays.asList(
            title,
            icon,
            mainImage,
            sponsored,
            description,
            ctaText
        )

        return AppstockNativeAdConfig.Builder()
            .setContextType(NativeContextType.SOCIAL_CENTRIC)
            .setPlacementType(NativePlacementType.CONTENT_FEED)
            .setContextSubType(NativeContextSubtype.GENERAL_SOCIAL)
            .setNativeEventTrackers(listOf(eventTracker))
            .setNativeAssets(assets)
            .build()
    }


    private fun createCustomView(ad: NativeAd) {
        val binding = ViewNativeAdAdMobBinding.inflate(LayoutInflater.from(this))

        binding.tvHeadline.setText(ad.headline)
        binding.tvBody.setText(ad.body)
        val icon = ad.icon
        if (icon != null) {
            binding.imgIco.setImageDrawable(icon.drawable)
        }
        if (!ad.images.isEmpty()) {
            binding.imgMedia.setMediaContent(ad.mediaContent)
        }

        val container: NativeAdView = binding.viewNativeWrapper
        container.headlineView = binding.tvHeadline
        container.bodyView = binding.tvBody
        container.iconView = binding.imgIco
        container.mediaView = binding.imgMedia
        container.setNativeAd(ad)

        containerForAd.addView(binding.getRoot())
    }


    companion object {
        private const val ADMOB_AD_UNIT_ID = "ca-app-pub-2844566227051243/6967515093"

        private fun createListener(): AdListener {
            return object : AdListener() {
                override fun onAdLoaded() {
                    Log.d(TAG, "Ad loaded successfully")
                }

                override fun onAdFailedToLoad(loadAdError: LoadAdError) {
                    Log.e(TAG, "Ad failed to load: $loadAdError")
                }
            }
        }
    }
}