/* 
 *  © 2024 Appstock 
 *  */

package com.appstock.demo.kotlin.activities.ads.inapp

import android.os.Bundle
import android.util.Log
import android.view.View
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import com.appstock.demo.kotlin.R
import com.appstock.demo.kotlin.activities.BaseAdActivity
import com.appstock.demo.kotlin.utils.ImageUtils
import com.appstock.sdk.api.adunits.AppstockNativeAdUnit
import com.appstock.sdk.api.adunits.AppstockNativeAdUnitEventListener
import com.appstock.sdk.api.adunits.nativead.request.*
import com.appstock.sdk.api.adunits.nativead.response.NativeEventTracker
import com.appstock.sdk.api.adunits.nativead.response.AppstockNativeAd
import com.appstock.sdk.api.adunits.nativead.response.AppstockNativeResult
import com.appstock.sdk.api.AdInfo
import java.util.*

class AppstockNativeActivity : BaseAdActivity() {

    private var adUnit: AppstockNativeAdUnit? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        createAd()
    }

    override fun onDestroy() {
        super.onDestroy()
        if (adUnit != null) {
            adUnit!!.destroy()
        }
    }


    private fun createAd() {
        // 1. Create AppstockNativeAdUnit
        adUnit = AppstockNativeAdUnit()

        // 2. Configure ad unit with native config
        adUnit?.setPlacementId(PLACEMENT_ID)
        adUnit?.setNativeAdConfig(createNativeConfig())

        // 3. Load ad
        adUnit?.loadAd { result: AppstockNativeResult ->
            val nativeAd = result.nativeAd
            if (nativeAd == null) {
                Log.e("AdExample", "Native ad is null: " + result.status)
                return@loadAd
            }

            Log.d(TAG, "Native ad loaded successfully")
            // 4. Create native view
            createNativeView(nativeAd)
        }
    }

    private fun createNativeConfig(): AppstockNativeAdConfig {
        val eventTrackingMethods = ArrayList(
            Arrays.asList(
                NativeEventTracker.EventTrackingMethod.IMAGE,
                NativeEventTracker.EventTrackingMethod.JS
            )
        )
        val eventTracker = NativeEventTracker(
            NativeEventTracker.EventType.IMPRESSION,
            eventTrackingMethods
        )

        val title = NativeTitleAsset()
        title.setLength(90)
        title.isRequired = true

        val icon = NativeImageAsset(20, 20, 20, 20)
        icon.imageType = NativeImageAsset.ImageType.ICON
        icon.isRequired = true

        val mainImage = NativeImageAsset(200, 200, 200, 200)
        mainImage.imageType = NativeImageAsset.ImageType.MAIN
        mainImage.isRequired = true

        val sponsored = NativeDataAsset()
        sponsored.len = 90
        sponsored.dataType = NativeDataAsset.DataType.SPONSORED
        sponsored.isRequired = true


        val description = NativeDataAsset()
        description.dataType = NativeDataAsset.DataType.DESC
        description.isRequired = true

        val ctaText = NativeDataAsset()
        ctaText.dataType = NativeDataAsset.DataType.CTATEXT
        ctaText.isRequired = true

        val assets = Arrays.asList(
            title,
            icon,
            mainImage,
            sponsored,
            description,
            ctaText
        )

        return AppstockNativeAdConfig.Builder()
            .setContextType(NativeContextType.SOCIAL_CENTRIC)
            .setPlacementType(NativePlacementType.CONTENT_FEED)
            .setContextSubType(NativeContextSubtype.GENERAL_SOCIAL)
            .setNativeEventTrackers(listOf(eventTracker))
            .setNativeAssets(assets)
            .build()
    }

    private fun createNativeView(ad: AppstockNativeAd) {
        val nativeContainer = View.inflate(this, R.layout.layout_native, null)

        val icon = nativeContainer.findViewById<ImageView>(R.id.imgIcon)
        ImageUtils.download(ad.iconUrl, icon)

        val title = nativeContainer.findViewById<TextView>(R.id.tvTitle)
        title.text = ad.title

        val image = nativeContainer.findViewById<ImageView>(R.id.imgImage)
        ImageUtils.download(ad.imageUrl, image)

        val description = nativeContainer.findViewById<TextView>(R.id.tvDesc)
        description.text = ad.description

        val cta = nativeContainer.findViewById<Button>(R.id.btnCta)
        cta.text = ad.callToAction

        containerForAd.addView(nativeContainer)

        ad.registerView(nativeContainer, arrayListOf(icon, title, image, description, cta), createListener())
    }

    companion object {
        private const val PLACEMENT_ID = "9"

        private fun createListener(): AppstockNativeAdUnitEventListener {
            return object : AppstockNativeAdUnitEventListener {
                override fun onAdImpression() {
                    // Called when ad displayed
                    Log.d(TAG, "Ad displayed on the screen")
                }

                override fun onAdClicked() {
                    // Called when ad clicked
                    Log.d(TAG, "Ad clicked")
                }

                override fun onAdExpired() {
                    // Called when ad expired
                    Log.d(TAG, "Ad expired")
                }
            }
        }
    }
}