/* © 2024 Appstock */

import UIKit

import AppstockSDK
import AppstockTopOnAdapter

import AnyThinkNative

fileprivate let placementID = "n66daf48383710"

final class TopOnNativeViewController: UIViewController {
    
    static var nib: String {
        "TopOnNativeViewController"
    }
    
    @IBOutlet weak var nativeContainerView: UIView!
    
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var bodyLabel: UILabel!
    @IBOutlet weak var sponsoredLabel: UILabel!
    @IBOutlet weak var mainImageView: UIImageView!
    @IBOutlet weak var iconView: UIImageView!
    @IBOutlet weak var callToActionLabel: UILabel!
    
    convenience init() {
        self.init(nibName: Self.nib, bundle: nil)
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        loadAd()
    }
    
    private func loadAd() {
        // 1. Configure the native parameters
        let image = AppstockNativeAssetImage(minimumWidth: 200, minimumHeight: 50, required: true)
        image.type = .Main
        
        let icon = AppstockNativeAssetImage(minimumWidth: 20, minimumHeight: 20, required: true)
        icon.type = .Icon
        
        let title = AppstockNativeAssetTitle(length: 90, required: true)
        let body = AppstockNativeAssetData(type: .description, required: true)
        let cta = AppstockNativeAssetData(type: .ctatext, required: true)
        let sponsored = AppstockNativeAssetData(type: .sponsored, required: true)
        
        let parameters = AppstockNativeParameters()
        parameters.assets = [title, icon, image, sponsored, body, cta]
        
        let eventTracker = AppstockNativeEventTracker(
            event: .Impression,
            methods: [.Image, .js]
        )
        
        parameters.eventtrackers = [eventTracker]
        parameters.context = .Social
        parameters.placementType = .FeedContent
        parameters.contextSubType = .Social
        
        // 2. Set up the extras
        let extra = [
            kAppstockNativeAssets: parameters
        ]
        
        // 3. Load the ad
        ATAdManager.shared().loadAD(
            withPlacementID: placementID,
            extra: extra,
            delegate: self
        )
    }
}

extension TopOnNativeViewController: ATAdLoadingDelegate {
    
    func didFinishLoadingAD(withPlacementID placementID: String!) {
        // 4. Get the offer
        guard let offer = ATAdManager.shared().getNativeAdOffer(withPlacementID: placementID) else {
            return
        }
        
        // 5. Create a config
        let config = ATNativeADConfiguration()
        
        // 6. Create a ATNativeADView
        let nativeADView = ATNativeADView(
            configuration: config,
            currentOffer: offer,
            placementID: placementID
        )
        
        nativeADView.delegate = self
        
        // 7. Register clickable views
        nativeADView.registerClickableViewArray([callToActionLabel as Any,
                                                 mainImageView as Any,
                                                 iconView as Any])
        
        // 8. Bind the views
        let info = ATNativePrepareInfo.load { [weak self] info in
            info.titleLabel = self?.titleLabel ?? UILabel()
            info.advertiserLabel = self?.sponsoredLabel ?? UILabel()
            info.textLabel = self?.bodyLabel ?? UILabel()
            info.ctaLabel = self?.callToActionLabel ?? UILabel()
            info.mainImageView = self?.mainImageView ?? UIImageView()
            info.iconImageView = self?.iconView ?? UIImageView()
        }
        
        nativeADView.prepare(with: info)
        
        // 9. Call renderer
        offer.renderer(
            with: config,
            selfRenderView: nil,
            nativeADView: nativeADView
        )
        
        view.addSubview(nativeADView)
    }
    
    func didFailToLoadAD(withPlacementID placementID: String!, error: (any Error)!) {
        print("Did fail to load ad: \(error.localizedDescription)")
    }
    
    func didFailToLoadADSource(
        withPlacementID placementID: String!,
        extra: [AnyHashable : Any]!,
        error: (any Error)!
    ) {
        print("Did fail to load ad source: \(error.localizedDescription)")
    }
    
    func didFailBiddingADSource(
        withPlacementID placementID: String!,
        extra: [AnyHashable : Any]!,
        error: (any Error)!
    ) {
        print("Did fail bidding ad source: \(error.localizedDescription)")
    }
    
    func didRevenue(forPlacementID placementID: String!, extra: [AnyHashable : Any]!) {}
    func didStartLoadingADSource(withPlacementID placementID: String!, extra: [AnyHashable : Any]!) {}
    func didFinishLoadingADSource(withPlacementID placementID: String!, extra: [AnyHashable : Any]!) {}
    func didStartBiddingADSource(withPlacementID placementID: String!, extra: [AnyHashable : Any]!) {}
    func didFinishBiddingADSource(withPlacementID placementID: String!, extra: [AnyHashable : Any]!) {}
}

extension TopOnNativeViewController: ATNativeADDelegate {
    
    func didShowNativeAd(
        in adView: ATNativeADView,
        placementID: String,
        extra: [AnyHashable : Any]
    ) {}
    
    func didClickNativeAd(
        in adView: ATNativeADView,
        placementID: String,
        extra: [AnyHashable : Any]
    ) {}
}
