/* © 2024 Appstock */

import UIKit

import AppstockSDK
import GoogleMobileAdsAppstockAdapter

import GoogleMobileAds

fileprivate let adUnitId = "ca-app-pub-2844566227051243/5269825003"

final class AdMobNativeView: GoogleMobileAds.NativeAdView {}

final class AdMobNativeViewController: UIViewController {
    
    static var nib: String {
        "AdMobNativeViewController"
    }
    
    @IBOutlet weak var titleLabel: UILabel!
    @IBOutlet weak var bodyLabel: UILabel!
    @IBOutlet weak var sponsoredLabel: UILabel!
    @IBOutlet weak var mainImageView: UIImageView!
    @IBOutlet weak var iconView: UIImageView!
    @IBOutlet weak var callToActionButton: UIButton!
    
    @IBOutlet weak var admobNativeView: AdMobNativeView!
    
	private var adLoader: GoogleMobileAds.AdLoader?
    
    convenience init() {
        self.init(nibName: Self.nib, bundle: nil)
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        // Set the subviews of the GoogleMobileAds.NativeAdView
        admobNativeView.iconView = iconView
        admobNativeView.imageView = mainImageView
        admobNativeView.headlineView = titleLabel
        admobNativeView.bodyView = bodyLabel
        admobNativeView.callToActionView = callToActionButton
        admobNativeView.advertiserView = sponsoredLabel
        
        loadAd()
    }
    
    private func loadAd() {
        // 1. Create a GoogleMobileAds.AdLoader
		adLoader = GoogleMobileAds.AdLoader(
            adUnitID: adUnitId,
            rootViewController: self,
            adTypes: [.native],
            options: []
        )
        
        // 2. Configure the GoogleMobileAds.AdLoader
        adLoader?.delegate = self
        
        // 3. Configure the native parameters
        let image = AppstockNativeAssetImage(minimumWidth: 200, minimumHeight: 50, required: true)
        image.type = .Main
        
        let icon = AppstockNativeAssetImage(minimumWidth: 20, minimumHeight: 20, required: true)
        icon.type = .Icon
        
        let title = AppstockNativeAssetTitle(length: 90, required: true)
        let body = AppstockNativeAssetData(type: .description, required: true)
        let cta = AppstockNativeAssetData(type: .ctatext, required: true)
        let sponsored = AppstockNativeAssetData(type: .sponsored, required: true)
        
        let parameters = AppstockNativeParameters()
        parameters.assets = [title, icon, image, sponsored, body, cta]
        
        let eventTracker = AppstockNativeEventTracker(
            event: .Impression,
            methods: [.Image, .js]
        )
        
        parameters.eventtrackers = [eventTracker]
        parameters.context = .Social
        parameters.placementType = .FeedContent
        parameters.contextSubType = .Social
        
        // 4. Create a AppstockGADExtras
        let extras = AppstockGADExtras(nativeParameters: parameters)
        
        // 5. Create a GoogleMobileAds.Request
		let request = GoogleMobileAds.Request()
        
        // 6. Register the AppstockGADExtras
        request.register(extras)
        
        // 7. Load the ad
        adLoader?.load(request)
    }
}

// MARK: - GoogleMobileAds.NativeAdLoaderDelegate

extension AdMobNativeViewController: GoogleMobileAds.NativeAdLoaderDelegate {
    
	func adLoader(_ adLoader: GoogleMobileAds.AdLoader, didReceive nativeAd: GoogleMobileAds.NativeAd) {
        // Set GoogleMobileAds.NativeAd in GoogleMobileAds.NativeAdView
        admobNativeView.nativeAd = nativeAd
        
        // 8. Render the ad
        titleLabel.text = nativeAd.headline
        bodyLabel.text = nativeAd.body
        
        mainImageView.setImage(
            from: nativeAd.images?.last?.imageURL?.absoluteString,
            placeholder: UIImage(systemName: "photo.artframe")
        )
        
        iconView.setImage(
            from: nativeAd.icon?.imageURL?.absoluteString,
            placeholder: UIImage(systemName: "photo.artframe")
        )
        
        callToActionButton.setTitle(nativeAd.callToAction, for: .normal)
        sponsoredLabel.text = nativeAd.advertiser
    }
    
	func adLoader(_ adLoader: GoogleMobileAds.AdLoader, didFailToReceiveAdWithError error: any Error) {
        print("Did fail to receive ad with error: \(error.localizedDescription)")
    }
}
