# Appstock SDK iOS - Interstitial

To load interstitial ads, you should create and configure the `AppstockInterstitialAdUnit` and call its `loadAd()` method.

*Swift*

```swift
private var interstitialAdUnit: AppstockInterstitialAdUnit!
 
private func loadAd() {
    // 1. Create a AppstockInterstitialAdUnit
    interstitialAdUnit = AppstockInterstitialAdUnit()
     
    // 2. Configure the AppstockInterstitialAdUnit
    interstitialAdUnit.placementID = placementID
    interstitialAdUnit.delegate = self
     
    // 3. Load the interstitial ad
    interstitialAdUnit.loadAd()
}
```

*Objective-C*

```objc
@property (nonatomic) AppstockInterstitialAdUnit * interstitialAdUnit;

- (void)loadAd {
    // 1. Create a AppstockInterstitialAdUnit
    self.interstitialAdUnit = [[AppstockInterstitialAdUnit alloc] init];
    
    // 2. Configure the AppstockInterstitialAdUnit
    self.interstitialAdUnit.placementID = self.placementID;
    self.interstitialAdUnit.delegate = self;
    
    // 3. Load the interstitial ad
    [self.interstitialAdUnit loadAd];
}
```

If you need to integrate **video** ads or **multiformat** ads, you should set the adFormats property to the respective value:        

*Swift*

```swift
// Make ad request for video ad
interstitialAdUnit.adFormats = [.video]
 
// Make ad request for both video and banner ads (default behaviour)
interstitialAdUnit.adFormats = [.video, .banner]
 
// Make ad request for banner ad 
interstitialAdUnit.adFormats = [.banner]
```

*Objective-C*

```objc
// Make ad request for video ad
interstitialAdUnit.adFormats = [NSSet setWithArray:@[AppstockAdFormat.video]];
 
// Make ad request for both video and banner ads (default behaviour)
interstitialAdUnit.adFormats = [NSSet setWithArray:@[AppstockAdFormat.video, AppstockAdFormat.banner]];
 
// Make ad request for banner ad
interstitialAdUnit.adFormats = [NSSet setWithArray:@[AppstockAdFormat.banner]];
```

You can check if the ad is ready to be shown by calling respective property:

*Swift*

```swift
if interstitialAdUnit.isReady {
    // Show the ad...
}
```

*Objective-C*

```objc
if (interstitialAdUnit.isReady) {
        
}
```

Once the ad is loaded, you can invoke the `show()` method at any appropriate point of the app flow to present the fullscreen ad. To know when the ad is loaded, you should implement `AppstockInterstitialAdUnitDelegate` protocol and subscribe to the ad events in its methods.

When the delegate’s method `interstitialDidReceiveAd` is called, it means that the SDK has successfully loaded the ad. Starting from this point, you can call the `show()` method to display the full-screen ad.

The ad’s basic properties can be accessed through the [AppstockAdInfo](./2.2-appstock-sdk-ios-utils.md#teqblazeadinfo) structure once the ad has been loaded. Currently AppstockSDK provides the ad price and later this object will be extended.

*Swift*

```swift 
extension AppstockBannerInterstitialViewController: 
AppstockInterstitialAdUnitDelegate {
     
    func interstitialDidReceiveAd(_ interstitial: AppstockInterstitialAdUnit, adInfo: AppstockAdInfo) {
        // Called when ad is loaded
         
        // Show the full screen ad
        if interstitialAdUnit.isReady {
            interstitial.show(from: self)
        }
    }
     
    func interstitial(
        _ interstitial: AppstockInterstitialAdUnit,
        didFailToReceiveAdWithError error: (any Error)?
    ) {
        // Called when Appstock SDK failed to load ad
        print("Did fail to receive ad with error: 
        \(String(describing: error?.localizedDescription))")
    }
     
    func interstitialWillPresentAd(_ interstitial: AppstockInterstitialAdUnit) {
        // Called when interstitial is about to be presented
    }
     
    func interstitialDidDismissAd(_ interstitial: AppstockInterstitialAdUnit) 
    {
        // Called when interstitial is dismissed
    }
     
    func interstitialDidClickAd(_ interstitial: AppstockInterstitialAdUnit) {
        // Called when interstitial was clicked
    }
     
    func interstitialWillLeaveApplication(_ interstitial: 
    AppstockInterstitialAdUnit) {
        // Called when the application is about to enter the background
    }
}
```

*Objective-C*

```objc
@interface AppstockBannerInterstitialViewController : UIViewController <AppstockInterstitialAdUnitDelegate>

@end

// ...

- (void)interstitial:(AppstockInterstitialAdUnit *)interstitial didFailToReceiveAdWithError:(NSError *)error {
    // Called when Appstock SDK failed to load ad
    NSLog(@"Did fail to receive ad with error: %@", error.localizedDescription);
}

- (void)interstitialDidReceiveAd:(AppstockInterstitialAdUnit *)interstitial adInfo:(AppstockAdInfo *)adInfo {
    // Called when ad is loaded
    [interstitial showFrom:self];
}

- (void)interstitialWillPresentAd:(AppstockInterstitialAdUnit *)interstitial {
    // Called when interstitial is about to be presented
}

- (void)interstitialDidDismissAd:(AppstockInterstitialAdUnit *)interstitial {
    // Called when interstitial is dismissed
}

- (void)interstitialDidClickAd:(AppstockInterstitialAdUnit *)interstitial {
    // Called when interstitial was clicked
}

- (void)interstitialWillLeaveApplication:(AppstockInterstitialAdUnit *)interstitial {
    // Called when the application is about to enter the background
}
```

### Rendering Controls

The following properties enable rendering customization of video interstitial ads.

| Property             | Description                                                                                                                                                     |
|----------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------|
| isMuted              | This option lets you switch the sound on or off during playback. Default is `false`.                                                                            |
| closeButtonArea      | This setting determines the percentage of the device screen that the close button should cover. Allowed range - `0...1`. Default value is `0.1`.                |
| closeButtonPosition  | This setting controls where the close button appears on the screen. Allowed values: `topLeft`, `topRight`. Other values will be ignored. Default is `topRight`. |
| skipButtonArea       | This setting determines the percentage of the device screen that the skip button should cover. Allowed range - `0...1`. Default value is `0.1`.                 |
| skipButtonPosition   | This control sets the position of the skip button. Allowed values: `topLeft`, `topRight`. Other values will be ignored. Default is `topLeft`.                   |
| skipDelay            | This setting determines the number of seconds after the start of playback before the skip or close button should appear. Default value is `10.0`.               |
| isSoundButtonVisible | This option switches on or off the visibility of the sound/mute button for users. Default value is `false`.                                                     |

Usage example: 

*Swift*

```swift
interstitialAdUnit.isMuted = true
interstitialAdUnit.closeButtonArea = 0.2
interstitialAdUnit.closeButtonPosition = .topRight
interstitialAdUnit.skipButtonArea = 0.2
interstitialAdUnit.skipButtonPosition = .topLeft
interstitialAdUnit.skipDelay = 15.0
interstitialAdUnit.isSoundButtonVisible = true
```

*Objective-C*

```objc
interstitialAdUnit.isMuted = YES;
interstitialAdUnit.closeButtonArea = 0.2;
interstitialAdUnit.closeButtonPosition = AppstockPositionTopRight;
interstitialAdUnit.skipButtonArea = 0.2;
interstitialAdUnit.skipButtonPosition = AppstockPositionTopLeft;
interstitialAdUnit.skipDelay = 15.0;
interstitialAdUnit.isSoundButtonVisible = YES;
```
