# Appstock SDK iOS - Native

To load a native ad, you should initialize and configure `AppstockNativeAdUnit` object and call the `loadAd()` method.

*Swift*

```swift
private var nativeAdUnit: AppstockNativeAdUnit!
private var nativeAd: AppstockNativeAd?
 
private func loadAd() {
    // 1. Create a AppstockNativeAdUnit
    nativeAdUnit = AppstockNativeAdUnit()
     
    // 2. Configure the AppstockNativeAdUnit
    nativeAdUnit.placementID = placementID
    let image = AppstockNativeAssetImage(minimumWidth: 200, 
    minimumHeight: 50, required: true)
    image.type = .Main
     
    let icon = AppstockNativeAssetImage(minimumWidth: 20, 
    minimumHeight: 20, required: true)
    icon.type = .Icon
     
    let title = AppstockNativeAssetTitle(length: 90, required: true)
    let body = AppstockNativeAssetData(type: .description, required: true)
    let cta = AppstockNativeAssetData(type: .ctatext, required: true)
    let sponsored = AppstockNativeAssetData(type: .sponsored, required: true)
     
    let parameters = AppstockNativeParameters()
    parameters.assets = [title, icon, image, sponsored, body, cta]
     
    let eventTracker = AppstockNativeEventTracker(
        event: .Impression,
        methods: [.Image, .js]
    )
     
    parameters.eventtrackers = [eventTracker]
    parameters.context = .Social
    parameters.placementType = .FeedContent
    parameters.contextSubType = .Social
     
    nativeAdUnit.parameters = parameters
     
    nativeAdUnit.loadAd { [weak self] ad, error in
        guard let self = self else {
            return
        }
         
        guard let ad = ad, error == nil else {
            return
        }
         
        self.nativeAd = ad
        self.nativeAd?.delegate = self
         
        // 3. Render the native ad
        self.titleLabel.text = ad.title
        self.bodyLabel.text = ad.text
        self.sponsoredLabel.text = ad.sponsoredBy
         
        self.mainImageView.setImage(from: ad.imageUrl, p
        laceholder: UIImage(systemName: "photo.artframe"))
        self.iconView.setImage(from: ad.iconUrl, 
        placeholder: UIImage(systemName: "photo.artframe"))
        
        self.callToActionButton.setTitle(ad.callToAction, for: .normal)
         
        self.nativeAd?.registerView(view: self.view, 
        clickableViews: [self.callToActionButton])

        // Use `AdInfo` to get information about the received bid.
        print("Bid price: \(String(describing: nativeAd?.adInfo?.price))")
    }
}
```

*Objective-C*

```objc
@property (nonatomic) AppstockNativeAdUnit * nativeAdUnit;
@property (nonatomic, nullable) AppstockNativeAd * nativeAd;

- (void)loadAd {
    // 1. Create a AppstockNativeAdUnit
    self.nativeAdUnit = [[AppstockNativeAdUnit alloc] init];
    
    // 2. Configure the AppstockNativeAdUnit
    self.nativeAdUnit.placementID = self.placementID;
    
    AppstockNativeAssetImage *image = [
        [AppstockNativeAssetImage alloc]
        initWithMinimumWidth:200
        minimumHeight:200
        required:true
    ];
    
    image.type = AppstockImageAsset.Main;
    
    AppstockNativeAssetImage *icon = [
        [AppstockNativeAssetImage alloc]
        initWithMinimumWidth:20
        minimumHeight:20
        required:true
    ];
    
    icon.type = AppstockImageAsset.Icon;
    
    AppstockNativeAssetTitle *title = [
        [AppstockNativeAssetTitle alloc]
        initWithLength:90
        required:true
    ];
    
    AppstockNativeAssetData *body = [
        [AppstockNativeAssetData alloc]
        initWithType:AppstockDataAssetDescription
        required:true
    ];
    
    AppstockNativeAssetData *cta = [
        [AppstockNativeAssetData alloc]
        initWithType:AppstockDataAssetCtatext
        required:true
    ];
    
    AppstockNativeAssetData *sponsored = [
        [AppstockNativeAssetData alloc]
        initWithType:AppstockDataAssetSponsored
        required:true
    ];
    
    AppstockNativeParameters * parameters = [AppstockNativeParameters new];
    parameters.assets = @[title, icon, image, sponsored, body, cta];
    
    AppstockNativeEventTracker * eventTracker = [
        [AppstockNativeEventTracker alloc]
        initWithEvent:AppstockEventType.Impression
        methods:@[AppstockEventTracking.Image, AppstockEventTracking.js]
    ];
    
    parameters.eventtrackers = @[eventTracker];
    parameters.context = AppstockContextType.Social;
    parameters.placementType = AppstockPlacementType.FeedContent;
    parameters.contextSubType = AppstockContextSubType.Social;
    
    self.nativeAdUnit.parameters = parameters;
    
    __weak typeof(self) weakSelf = self;
    [self.nativeAdUnit loadAdWithCompletion:^(AppstockNativeAd * _Nullable ad, NSError * _Nullable error) {
        if (error != nil || ad == nil) {
            return;
        }
        
        weakSelf.nativeAd = ad;
        weakSelf.nativeAd.delegate = self;
        
        weakSelf.titleLabel.text = ad.title;
        weakSelf.bodyLabel.text = ad.text;
        weakSelf.sponsoredLabel.text = ad.sponsoredBy;
        
        [weakSelf.mainImageView setImageFromURLString:ad.imageUrl
         placeholder:[UIImage systemImageNamed:@"photo.artframe"]];
        [weakSelf.iconView setImageFromURLString:ad.iconUrl
        placeholder:[UIImage systemImageNamed:@"photo.artframe"]];
        [weakSelf.callToActionButton 
        setTitle:ad.callToAction  forState:UIControlStateNormal];

        // Use `AdInfo` to get information about the received bid.
        NSLog(@"Bid price: %@", weakSelf.nativeAd.adInfo.price ?: @"(null)");
    }];
}
```

You can configure the native assets by setting up `parameters` property. Here is a brief description of `AppstockNativeParameters`:

- **`assets`** - an array of assets associated with the native ad.

- **`eventtrackers`** - an array of event trackers used for tracking native ad events.

- **`version`** - the version of the native parameters. Default is `"1.2"`.

- **`context`** - the context type for the native ad (e.g., content, social).

- **`contextSubType`** - a more	detailed context in which the ad	appears.

- **`placementType`** - the design/format/layout of the ad unit	being	offered.

- **`placementCount`** - the	number of identical placements in	this layout. Default is `1`.

- **`sequence`** - the sequence number of the ad in a series. Default is `0`.

- **`asseturlsupport`** - whether the supply source / impression impression	supports	returning an assetsurl	instead	of	an	asset	object. Default is `0` (unsupported).

- **`durlsupport`**  - whether the supply source / impression	supports	returning a	dco url instead of an asset object. Default is `0` (unsupported).		

- **`privacy`**  - set	to	1 when the native ad support buyer-specific	privacy notice. Default is `0`.

- **`ext`**  - a dictionary to hold any additional data as key-value pairs.

Here is a brief description of available assets:

| Class/Enum                    | Type        | Name                | Description                                                                                       |
|-------------------------------|-------------|---------------------|---------------------------------------------------------------------------------------------------|
| `AppstockNativeAssetTitle`    | Class       |                     | A subclass representing a title asset in a native ad.                                             |
|                               | Property    | `ext`               | An optional extension for additional data.                                                        |
|                               | Property    | `length`            | The length of the title.                                                                          |
| `AppstockNativeAssetImage`    | Class       |                     | A subclass representing an image asset in a native ad.                                            |
|                               | Property    | `type`              | The type of image asset (e.g., icon, main image).                                                 |
|                               | Property    | `width`             | The width of the image.                                                                           |
|                               | Property    | `widthMin`          | The minimum width of the image.                                                                   |
|                               | Property    | `height`            | The height of the image.                                                                          |
|                               | Property    | `heightMin`         | The minimum height of the image.                                                                  |
|                               | Property    | `mimes`             | An array of supported MIME types for the image.                                                   |
|                               | Property    | `ext`               | An optional extension for additional data.                                                        |
| `AppstockNativeAssetData`     | Class       |                     | A subclass representing a data asset in a native ad.                                              |
|                               | Property    | `length`            | The length of the data string.                                                                    |
|                               | Property    | `ext`               | An optional extension for additional data.                                                        |
|                               | Property    | `type`              | The type of data asset (e.g., sponsored, description).                                            |
| `AppstockImageAsset`          | Class       |                     | A class representing different types of image assets in the Appstock SDK.                         |
|                               | Property    | `Icon`              | Represents an icon image asset.                                                                   |
|                               | Property    | `Main`              | Represents a main image asset.                                                                    |
|                               | Property    | `Custom`            | Represents a custom image asset.                                                                  |
| `AppstockDataAsset`           | Enum        |                     | An enum representing different types of data assets in the Appstock SDK.                          |
|                               | Case        | `sponsored`         | Represents sponsored content.                                                                     |
|                               | Case        | `description`       | Represents a description.                                                                         |
|                               | Case        | `rating`            | Represents a rating.                                                                              |
|                               | Case        | `likes`             | Represents likes.                                                                                 |
|                               | Case        | `downloads`         | Represents download count.                                                                        |
|                               | Case        | `price`             | Represents the price.                                                                             |
|                               | Case        | `saleprice`         | Represents a sale price.                                                                          |
|                               | Case        | `phone`             | Represents a phone number.                                                                        |
|                               | Case        | `address`           | Represents an address.                                                                            |
|                               | Case        | `description2`      | Represents a secondary description.                                                               |
|                               | Case        | `displayurl`        | Represents a display URL.                                                                         |
|                               | Case        | `ctatext`           | Represents call-to-action text.                                                                   |
|                               | Case        | `Custom`            | Represents a custom data asset.                                                                   |
|                               | Method      | `exchangeID`        | Returns or sets the exchange ID for the `Custom` case.                         

You can also specify what type of event tracking is supported. For that you need to set `eventtrackers` property. Here is a brief description of available types:

| Class/Enum                    | Type        | Name                | Description                                                                                       |
|-------------------------------|-------------|---------------------|---------------------------------------------------------------------------------------------------|
| `AppstockNativeEventTracker`  | Class       |                     | A class representing event trackers for native ads.                                               |
|                               | Property    | `event`             | The type of event to be tracked (e.g., impression, viewable impression).                          |
|                               | Property    | `methods`           | An array of tracking methods used for the event.                                                  |
|                               | Property    | `ext`               | An optional extension for additional data.                                                        |
| `AppstockEventType`           | Class       |                     | A class representing different event types that can be tracked.                                   |
|                               | Property    | `Impression`        | Represents an impression event.                                                                   |
|                               | Property    | `ViewableImpression50` | Represents a 50% viewable impression event.                                                    |
|                               | Property    | `ViewableImpression100` | Represents a 100% viewable impression event.                                                 |
|                               | Property    | `ViewableVideoImpression50` | Represents a 50% viewable video impression event.                                         |
|                               | Property    | `Custom`            | Represents a custom event type.                                                                   |
| `AppstockEventTracking`       | Class       |                     | A class representing different methods of event tracking.                                         |
|                               | Property    | `Image`             | Represents image-based event tracking.                                                            |
|                               | Property    | `js`                | Represents JavaScript-based event tracking.                                                       |
|                               | Property    | `Custom`            | Represents a custom tracking method.                                                              |


Once the ad is loaded, the SDK provides you with a `AppstockNativeAd` object in the callback of the `loadAd()` method. This object contains ad assets that you should apply to the native ad layout.

If you need to manage stages of the ad lifecycle you should implement the `AppstockNativeAdDelegate` protocol.

The ad’s basic properties can be accessed through the [AppstockAdInfo](./2.2-appstock-sdk-ios-utils.md#teqblazeadinfo) structure once the ad has been loaded. Currently AppstockSDK provides the ad price and later this object will be extended.

*Swift*

```swift
extension AppstockNativeViewController: AppstockNativeAdDelegate {
     
    func adDidExpire(ad: AppstockNativeAd) {
        // Called when the ad expired
    }
     
    func adWasClicked(ad: AppstockNativeAd) {
        // Called when the ad was clicked
    }
     
    func adDidLogImpression(ad: AppstockNativeAd) {
        // Called when the impression was logged
    }
}
```

*Objective-C*

```objc
@interface AppstockNativeViewController : UIViewController<AppstockNativeAdDelegate>

@end

// ...

- (void)adDidExpireWithAd:(AppstockNativeAd *)ad {
    // Called when the ad expired
}

- (void)adWasClickedWithAd:(AppstockNativeAd *)ad {
    // Called when the ad was clicked
}

- (void)adDidLogImpressionWithAd:(AppstockNativeAd *)ad {
    // Called when the impression was logged
}
```

If you need ORTB native request object, you can use `getNativeRequestObject` method for that. It returns a dictionary with request configuration.

*Swift*

```swift
let request = adUnit.getNativeRequestObject()
```

*Objective-C*

```objc
NSDictionary * request = [self.nativeAdUnit getNativeRequestObject];
```