# Appstock SDK iOS - Rewarded

To load rewarded ads, you should create and configure the `AppstockRewardedAdUnit` and call its `loadAd()` method.

*Swift*

```swift
private var rewardedAdUnit: AppstockRewardedAdUnit!
 
private func loadAd() {
    // 1. Create a AppstockRewardedAdUnit
    rewardedAdUnit = AppstockRewardedAdUnit()
     
    // 2. Configure the AppstockRewardedAdUnit
    rewardedAdUnit.placementID = placementID
    rewardedAdUnit.delegate = self
     
    // 3. Load the rewarded ad
    rewardedAdUnit.loadAd()
}
```

*Objective-C*

```objc
@property (nonatomic) AppstockRewardedAdUnit * rewardedAdUnit;

- (void)loadAd {
    // 1. Create a AppstockRewardedAdUnit
    self.rewardedAdUnit = [[AppstockRewardedAdUnit alloc] init];
    
    // 2. Configure the AppstockRewardedAdUnit
    self.rewardedAdUnit.placementID = self.placementID;
    self.rewardedAdUnit.delegate = self;
    
    // 3. Load the rewarded ad
    [self.rewardedAdUnit loadAd];
}
```

If you need to integrate **video** ads or **multiformat** ads, you should set the adFormats property to the respective value:        

*Swift*

```swift
// Make ad request for video ad
rewardedAdUnit.adFormats = [.video]
 
// Make ad request for both video and banner ads (default behaviour)
rewardedAdUnit.adFormats = [.video, .banner]
 
// Make ad request for banner ad
rewardedAdUnit.adFormats = [.banner]
```

*Objective-C*

```objc
// Make ad request for video ad
rewardedAdUnit.adFormats = [NSSet setWithArray:@[AppstockAdFormat.video]];
 
// Make ad request for both video and banner ads (default behaviour)
rewardedAdUnit.adFormats = [NSSet setWithArray:@[AppstockAdFormat.video, AppstockAdFormat.banner]];
 
// Make ad request for banner ad
rewardedAdUnit.adFormats = [NSSet setWithArray:@[AppstockAdFormat.banner]];
```

You can check if the ad is ready to be shown by calling respective property:

*Swift*

```swift
if rewardedAdUnit.isReady {
    // Show the ad...
}
```

*Objective-C*

```objc
if (rewardedAdUnit.isReady) {
        
}
```

Once the ad is loaded, you can invoke the `show()` method at any appropriate point of the app flow to present the fullscreen ad. To know when the ad is loaded, you should implement `AppstockRewardedAdUnitDelegate` protocol and subscribe to the ad events in its methods.

When the delegate’s method `rewardedAdDidReceiveAd` is called, it means that the SDK has successfully loaded the ad. Starting from this point, you can call the `show()` method to display the full-screen ad.

The ad’s basic properties can be accessed through the [AppstockAdInfo](./2.2-appstock-sdk-ios-utils.md#teqblazeadinfo) structure once the ad has been loaded. Currently AppstockSDK provides the ad price and later this object will be extended.

*Swift*

```swift 
extension AppstockBannerRewardedViewController: 
AppstockRewardedAdUnitDelegate {
     
    func rewardedAdDidReceiveAd(_ rewardedAd: AppstockRewardedAdUnit, adInfo: AppstockAdInfo) {
        // Called when ad is loaded
         
        // Show the full screen ad
        if rewardedAd.isReady {
            rewardedAd.show(from: self)
        }
    }
     
    func rewardedAd(
        _ rewardedAd: AppstockRewardedAdUnit,
        didFailToReceiveAdWithError error: Error?
    ) {
        // Called when Appstock SDK failed to load ad
        print("Did fail to receive ad with error: 
        \(String(describing: error?.localizedDescription))")
    }
     
    func rewardedAdWillPresentAd(_ rewardedAd: AppstockRewardedAdUnit) {
        // Called when rewarded ad is about to be presented
    }
     
    func rewardedAdDidDismissAd(_ rewardedAd: AppstockRewardedAdUnit) {
        // Called when rewarded ad is dismissed
    }
     
    func rewardedAdDidClickAd(_ rewardedAd: AppstockRewardedAdUnit) {
        // Called when rewarded ad was clicked
    }
     
    func rewardedAdWillLeaveApplication(_ rewardedAd: AppstockRewardedAdUnit) {
        // Called when the application is about to enter the background
    }
    
    func rewardedAdUserDidEarnReward(_ rewardedAd: AppstockRewardedAdUnit, reward: AppstockReward) {
        // Called when the reward was granted to user
    }
}
```

*Objective-C*

```objc
@interface AppstockBannerRewardedViewController : UIViewController <AppstockRewardedAdUnitDelegate>

@end

// ...

- (void)rewardedAdDidReceiveAd:(AppstockRewardedAdUnit *)rewardedAd adInfo:(AppstockAdInfo *)adInfo {
    // Called when ad is loaded
    [rewardedAd showFrom:self];
}

- (void)rewardedAd:(AppstockRewardedAdUnit *)rewardedAd didFailToReceiveAdWithError:(NSError *)error {
    // Called when Appstock SDK failed to load ad
    NSLog(@"Did fail to receive ad with error: %@", error.localizedDescription);
}

- (void)rewardedAdWillPresentAd:(AppstockRewardedAdUnit *)rewardedAd {
    // Called when rewarded ad is about to be presented
}

- (void)rewardedAdDidDismissAd:(AppstockRewardedAdUnit *)rewardedAd {
    // Called when rewarded ad is dismissed
}

- (void)rewardedAdDidClickAd:(AppstockRewardedAdUnit *)rewardedAd {
    // Called when rewarded ad was clicked
}

- (void)rewardedAdWillLeaveApplication:(AppstockRewardedAdUnit *)rewardedAd {
    // Called when the application is about to enter the background
}

- (void)rewardedAdUserDidEarnReward:(AppstockRewardedAdUnit *)rewardedAd reward:(AppstockReward *)reward {
    // Called when the reward was granted to user
}
```

### Rendering Controls

The following properties enable rendering customization of video rewarded ads.

| Property             | Description                                                                                                                                                     |
|----------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------|
| isMuted              | This option lets you switch the sound on or off during playback. Default is `false`.                                                                            |
| closeButtonArea      | This setting determines the percentage of the device screen that the close button should cover. Allowed range - `0...1`. Default value is `0.1`.                |
| closeButtonPosition  | This setting controls where the close button appears on the screen. Allowed values: `topLeft`, `topRight`. Other values will be ignored. Default is `topRight`. |
| isSoundButtonVisible | This option switches on or off the visibility of the sound/mute button for users. Default value is `false`.                                                     |

Usage example: 

*Swift*

```swift
rewardedAdUnit.isMuted = true
rewardedAdUnit.closeButtonArea = 0.2
rewardedAdUnit.closeButtonPosition = .topRight
rewardedAdUnit.isSoundButtonVisible = true
```

*Objective-C*

```objc
rewardedAdUnit.isMuted = YES;
rewardedAdUnit.closeButtonArea = 0.2;
rewardedAdUnit.closeButtonPosition = AppstockPositionTopRight;
rewardedAdUnit.isSoundButtonVisible = YES;
```
