# Appstock SDK iOS - Banner

To load a banner ad, create a `AppstockAdView` object, configure it, add it to the view hierarchy, and call its `loadAd()` method.

*Swift*

```swift
private var adView: AppstockAdView!
 
private func loadAd() {
    // 1. Create a AppstockAdView
    adView = AppstockAdView(
        frame: CGRect(origin: .zero, size: CGSize(width: 300, height: 250))
    )
     
    // 2. Configure the AppstockAdView
    adView.placementID = placementID
    adView.delegate = self
     
    // Add Appstock ad view to the app UI
    containerAdView.addSubview(adView)
     
    // 3. Load the ad
    adView.loadAd()
}
```

*Objective-C*

```objc
@property (nonatomic) AppstockAdView * adView;

- (void)loadAd {
    // 1. Create a AppstockAdView
    self.adView = [[AppstockAdView alloc] initWithFrame:CGRectMake(0, 0, 300, 250)];
    
    // 2. Configure the AppstockAdView
    self.adView.placementID = self.placementID;
    self.adView.delegate = self;
    
    // Add Appstock ad view to the app UI
    [self.containerAdView addSubview:self.adView];
    
    // 3. Load the ad
    [self.adView loadAd];
}
```

The `AppstockAdView` should be provided with one of the required configuration properties:

- **placementID** - unique placement identifier generated on the Appstock platform's UI;
- **endpointID** - unique endpoint identifier generated on the Appstock platform's UI.

Which one to use depends on your type of Appstock account.

You should also provide `CGRect` value for ad view to initialize `UIView`.

If you need to integrate video ads, you can also use the `AppstockAdView` object in the same way as for banner ads. The single required change is you should explicitly set the ad format via the respective property:

*Swift*

```swift
adView.adFormat = .video
```

*Objective-C*

```objc
self.adView.adFormat = AppstockAdFormat.video;
```

Once it is done, the TeqBlzae SDK will make ad requests for video placement and render the respective creatives.

You can optionally subscribe to the ad’s lifecycle events by implementing the `AppstockAdViewDelegate` protocol:

*Swift*

```swift
extension BannerAdViewController: AppstockAdViewDelegate {
    
    func adViewPresentationController() -> UIViewController? {
        // View controller used by SDK for presenting modal.
        // Usual implementation may simply return self, 
        // if it is view controller class.
        self
    }
     
    func adView(_ adView: AppstockAdView, didFailToReceiveAdWith error: any Error) {
        // Called when SDK failed to load ad
        print("Did fail to receive ad with error: \(error.localizedDescription)")
    }
     
    func adView(_ adView: AppstockAdView, didReceiveAdWithAdSize adSize: CGSize, adInfo: AppstockAdInfo) {
        // Called when ad is loaded
    }
     
    func adViewWillPresentModal(_ adView: AppstockAdView) {
        // Called when modal is about to be presented
    }
     
    func adViewDidDismissModal(_ adView: AppstockAdView) {
        // Called when modal is dismissed
    }
     
    func adViewWillLeaveApplication(_ adView: AppstockAdView) {
        // Called when the application is about to enter the background
    }
}
```

*Objective-C*

```objc
@interface AppstockBannerAdViewController : UIViewController <AppstockAdViewDelegate>

@end

// ...

- (UIViewController *)adViewPresentationController {
    // View controller used by SDK for presenting modal.
    // Usual implementation may simply return self, 
    // if it is view controller class.
    return self;
}

- (void)adView:(AppstockAdView *)adView didFailToReceiveAdWith:
(NSError *)error {
    // Called when Appstock SDK failed to load ad
    NSLog(@"Did fail to receive ad with error: %@", error.localizedDescription);
}

- (void)adView:(AppstockAdView *)adView 
didReceiveAdWithAdSize:(CGSize)adSize adInfo:(AppstockAdInfo *)adInfo {
    // Called when ad is loaded
}

- (void)adViewWillPresentModal:(AppstockAdView *)adView {
    // Called when modal is about to be presented
}

- (void)adViewDidDismissModal:(AppstockAdView *)adView {
    // Called when modal is dismissed
}

- (void)adViewWillLeaveApplication:(AppstockAdView *)adView {
    // Called when the application is about to enter the background
}
```

Once the ad is loaded you can utilize it's basic properties inspecting [AppstockAdInfo](./2.2-appstock-sdk-ios-utils.md#teqblazeadinfo) structure. Currently AppstockSDK provides the ad price and later this object will be extended.


The `refreshInterval` property controls the frequency of automatic ad refreshes. This interval is set in seconds and dictates how often a new ad request is made after the current ad is displayed.

*Swift*

```swift
adView.refreshInterval = 30.0
```

*Objective-C*

```objc
adView.refreshInterval = 30.0;
```

You can stop auto refresh by calling respective method:

*Swift*

```swift
adView.stopAutoRefresh()
```

*Objective-C*

```objc
[adView stopAutoRefresh];
```

You can also set `adPosition` property to specify the position of the ad on the screen and corresponding value will be sent in `bidRequest.imp[].banner.pos` ORTB field during bid request.

*Swift*

```swift
adView.adPosition = .footer
```

*Objective-C*

```objc
adView.adPostion = AppstockAdPositionFooter;
```