/* © 2024 Appstock */

/*   Copyright 2018-2021 Prebid.org, Inc.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

/*
 CHANGE NOTICE: The file content was modified to rebrand the work, adapt the product for the new use case, and change the distribution license.
 The patch to the original code is available in the Appstock repository.
 All changes after applying the patch are made under the new license and copyrights mentioned at the beginning of this file.
*/

//Superclass
#import <Foundation/Foundation.h>

//Protocols
#import "APSCreativeViewDelegate.h"
#import "APSVoidBlock.h"

//Classes
@class UIView;
@class UIViewController;

@protocol APSCreativeResolutionDelegate;
@class APSCreativeModel;
@class APSInterstitialDisplayProperties;
@class APSModalManager;
@class APSModalState;
@class APSTransaction;
@class APSEventManager;
@class APSOpenMeasurementSession;
@class APSDownloadDataHelper;
@class APSCreativeViewabilityTracker;
@class APSViewExposure;

NS_ASSUME_NONNULL_BEGIN

/**
 *  `APSAbstractCreative`'s purpose is a bundling of a model and a view. It acts as an adapter between
 *  the view and the SDK, it's essentially the C in MVC.
 *
 *  All `Creatives` must conform to this protocol. Each creative has-a model which contains the
 *  creative info, and must implement a few methods for handling display of the creative.
 */
@interface APSAbstractCreative : NSObject

@property (nonatomic, weak, readonly, nullable) APSTransaction *transaction;
@property (nonatomic, strong, nullable) APSCreativeModel *creativeModel;
@property (nonatomic, readonly, nonnull) APSEventManager *eventManager;
@property (nonatomic, strong, nullable) UIView *view;
@property (nonatomic, assign) BOOL clickthroughVisible;
@property (nonatomic, strong, nullable) APSModalManager *modalManager;
@property (nonatomic, strong, nonnull) dispatch_queue_t dispatchQueue;
@property (nonatomic, strong, nullable) APSCreativeViewabilityTracker *viewabilityTracker;
@property (nonatomic, copy, nullable, readonly) APSVoidBlock dismissInterstitialModalState;
@property BOOL isDownloaded;

// Indicates whether creative is opened with user action (expanded, clickthrough showed ...) or not
// Note that subclasses provide specific implementation.
@property (nonatomic, readonly) BOOL isOpened;

// The time that that the ad is displayed (i.e. before its refreshed).
// Note that subclasses provide specific implementation.
@property (nonatomic, readonly, nullable)NSNumber *displayInterval;

@property (nonatomic, weak, nullable) id<APSCreativeResolutionDelegate> creativeResolutionDelegate;
@property (nonatomic, weak, nullable) id<APSCreativeViewDelegate> creativeViewDelegate;
@property (nonatomic, weak, nullable) UIViewController* viewControllerForPresentingModals;

@property (nonatomic, readonly, getter=isMuted) BOOL muted;

- (instancetype)init NS_UNAVAILABLE;
- (instancetype)initWithCreativeModel:(APSCreativeModel *)creativeModel
                                  transaction:(APSTransaction *)transaction NS_DESIGNATED_INITIALIZER;

- (void)setupView;
- (void)displayWithRootViewController:(UIViewController *)viewController;
- (void)showAsInterstitialFromRootViewController:(UIViewController *)viewController displayProperties:(APSInterstitialDisplayProperties *)displayProperties;
- (void)handleClickthrough:(NSURL *)url;

//Resolution
- (void)onResolutionCompleted;
- (void)onResolutionFailed:(NSError *)error;

//Open Measurement
- (void)createOpenMeasurementSession;

- (void)pause;
- (void)resume;
- (void)mute;
- (void)unmute;

//Modal Manager Events
- (void)modalManagerDidFinishPop:(APSModalState *)state;
- (void)modalManagerDidLeaveApp:(APSModalState *)state;

- (void)onViewabilityChanged:(BOOL)viewable viewExposure:(APSViewExposure *)viewExposure;

@end

NS_ASSUME_NONNULL_END
